""" opengl_utils.py """
from OpenGL.GL import *


class OpenglUtils():
    """ methods to load, compile and link shaders """
    @staticmethod
    def initialise_shader(s_code, s_type):
        """ set up shader """
        s_ref = glCreateShader(s_type)
        glShaderSource(s_ref, '#version 460\n' + s_code)
        glCompileShader(s_ref)

        success = glGetShaderiv(s_ref, GL_COMPILE_STATUS)
        if not success:
            log_error = glGetShaderInfoLog(s_ref)
            glDeleteShader(s_ref)
            msg = '\n' + log_error.decode('utf-8')
            raise RuntimeError(msg)
        return s_ref

    @staticmethod
    def initialise_program(vs_code, fs_code):
        """ set up program """
        vs_ref = OpenglUtils.initialise_shader(vs_code, GL_VERTEX_SHADER)
        fs_ref = OpenglUtils.initialise_shader(fs_code, GL_FRAGMENT_SHADER)

        p_ref = glCreateProgram()
        glAttachShader(p_ref, vs_ref)
        glAttachShader(p_ref, fs_ref)
        glLinkProgram(p_ref)

        success = glGetProgramiv(p_ref, GL_LINK_STATUS)
        if not success:
            log_error = glGetProgramInfoLog(p_ref)
            glDeleteProgram(p_ref)
            msg = '\n' + log_error.decode('utf-8')
            raise RuntimeError(msg)
        return p_ref

    @staticmethod
    def system_info():
        """ display opengl information """
        info = {'Vendor': GL_VENDOR,
                'Renderer': GL_RENDERER,
                'OpenGL version': GL_VERSION,
                'GLSL version': GL_SHADING_LANGUAGE_VERSION}
        for key, value in info.items():
            print(key + ': ' + glGetString(value).decode('utf-8'))
