""" base.py """
import pyglet
from core.input import Input


class Base():
    """ graphics framework base class """
    def __init__(self, screen_size=(512, 512), caption='Warrior'):
        self.screen = pyglet.window.Window(
                visible=False,
                config=pyglet.gl.Config(double_buffer=True,
                                        sample_buffers=1,
                                        samples=4))
        self.screen.set_size(*screen_size)
        self.screen.set_caption(caption)
        self.screen.set_icon(
                pyglet.image.load('core/icons/warrior_32x32.png'),
                pyglet.image.load('core/icons/warrior_48x48.png'),
                pyglet.image.load('core/icons/warrior_72x72.png'),
                pyglet.image.load('core/icons/warrior_128x128.png'),
                pyglet.image.load('core/icons/warrior_256x256.png'))

        self.input = Input()
        self.locals = {}
        self.elapsed_time = 0.0

    def initialise(self):
        """ implemented by application """
        print('Initialising program ...')

    def finalise(self):
        """ implemented by application """
        print('... finalising program.')

    def update_base(self, delta):
        """ always called during run """
        self.elapsed_time += delta
        if self.elapsed_time > 3600.0:
            self.elapsed_time -= 3600.0
        self.update(delta)

    def update(self, delta):
        """ implemented by application """

    def draw(self):
        """ implemented by application """

    def run(self):
        """ let's get started! """
        @self.screen.event
        def on_key_press(symbol, modifiers):
            self.input.update_on_key_press(symbol, modifiers)
            if self.input.quit:
                self.screen.close()

        @self.screen.event
        def on_key_release(symbol, modifiers):
            self.input.update_on_key_release(symbol, modifiers)

        @self.screen.event
        def on_draw():
            self.draw()

        pyglet.clock.schedule_interval(self.update_base, 0.01)
        self.screen.set_visible()

        self.initialise()
        pyglet.app.run()
        self.finalise()
